import { Response, NextFunction } from 'express';
import { supabase } from '@/config/supabase';
import { AuthenticatedRequest } from '@/types';

/**
 * Middleware to authenticate user via Supabase token
 * Extracts token from Authorization header and verifies with Supabase
 */
export const authenticateUser = async (
  req: AuthenticatedRequest,
  res: Response,
  next: NextFunction
): Promise<void> => {
  try {
    // Get token from Authorization header
    const authHeader = req.headers.authorization;
    if (!authHeader || !authHeader.startsWith('Bearer ')) {
      res.status(401).json({
        success: false,
        error: 'Access token required'
      });
      return;
    }

    const token = authHeader.substring(7); // Remove 'Bearer ' prefix

    // Verify token with Supabase
    const { data: { user }, error } = await supabase.auth.getUser(token);

    if (error || !user) {
      res.status(401).json({
        success: false,
        error: 'Invalid or expired token'
      });
      return;
    }

    // Attach user to request
    req.user = user;
    next();
  } catch (error) {
    console.error('Authentication error:', error);
    res.status(500).json({
      success: false,
      error: 'Authentication failed'
    });
  }
};

/**
 * Middleware to require specific user role
 * Must be used after authenticateUser middleware
 */
export const requireRole = (allowedRoles: string[]) => {
  return (req: AuthenticatedRequest, res: Response, next: NextFunction): void => {
    if (!req.user) {
      res.status(401).json({
        success: false,
        error: 'Authentication required'
      });
      return;
    }

    // Check if user has required role
    const userRole = req.user.user_metadata?.role || 'vendor';
    
    if (!allowedRoles.includes(userRole)) {
      res.status(403).json({
        success: false,
        error: 'Insufficient permissions'
      });
      return;
    }

    next();
  };
};

/**
 * Middleware to require vendor role specifically
 */
export const requireVendor = requireRole(['vendor', 'admin', 'staff', 'superadmin']);

/**
 * Middleware to require admin role specifically
 */
export const requireAdmin = requireRole(['admin', 'superadmin']);

/**
 * Middleware to require super admin role specifically
 */
export const requireSuperAdmin = requireRole(['superadmin']); 